/*=============================================================================*
 *  Copyright 2006 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================*/

package org.apache.muse.test.wsrf;

import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.Date;

import javax.xml.namespace.QName;

import org.apache.muse.util.xml.XPathUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.apache.muse.ws.resource.lifetime.WsrlConstants;
import org.apache.muse.ws.resource.remote.WsResourceClient;
import org.apache.muse.ws.resource.sg.remote.ServiceGroupClient;

/**
 * 
 * WsrfTestClient is a simple exercise all of the WSRF specs and 
 * capabilities. It is not a comprehensive unit test, it is only 
 * meant to show that the WSRF capabilities are stable and that 
 * the WSRF-using application is deployed correctly. You can 
 * comment out any of the sub-tests in order to make the output 
 * more readable.
 *
 * @author Dan Jemiolo (danjemiolo)
 *
 */

public class WsrfTestClient
{    
    public static URI getLocalAddress(String contextPath, int port) 
        throws UnknownHostException
    {
        String ip = InetAddress.getLocalHost().getHostAddress();
        
        StringBuffer address = new StringBuffer();
        address.append("http://");
        address.append(ip);
        address.append(':');
        address.append(port);
        
        if (contextPath.charAt(0) != '/')
            address.append('/');
            
        address.append(contextPath);
        
        return URI.create(address.toString());
    }
    
    public static void destroy(WsResourceClient client) 
        throws Exception
    {
        client.destroy();
    }
    
    public static void setTerminationTime(WsResourceClient client) 
        throws Exception
    {
        Date now = new Date();
        long future = now.getTime() + 10000;
        client.setTerminationTime(new Date(future));
    }
    
    public static void getResourceProperty(WsResourceClient client) 
        throws Exception
    {
        client.getPropertyAsObject(WsrlConstants.CURRENT_TIME_QNAME, Date.class);
    }
    
    public static void getResourcePropertyDocument(WsResourceClient client)
        throws Exception
    {
        client.getResourcePropertyDocument();
    }
    
    public static void getMultipleResourceProperties(WsResourceClient client) 
        throws Exception
    {
        QName[] names = new QName[]{
            WsrlConstants.CURRENT_TIME_QNAME, WsrlConstants.TERMINATION_TIME_QNAME 
        };
        client.getMultipleResourceProperties(names);
    }
    
    public static void queryResourceProperties(WsResourceClient client)
        throws Exception
    {
        String query = "/*/*";
        String dialect = XPathUtils.NAMESPACE_URI;
        client.queryResourceProperties(query, dialect);
    }
    
    public static void setResourceProperties(WsResourceClient client)
        throws Exception
    {
        QName name = new QName("http://ws.apache.org/muse/test/wsrf", "MyProperty", "tns");
        
        Object[] values = new Object[]{ "Hello!", "Goodbye!" };
        client.insertResourceProperty(name, values);
        
        String query = "/*/tns:MyProperty";
        String dialect = XPathUtils.NAMESPACE_URI;
        client.queryResourceProperties(query, dialect);
        
        values = new Object[]{ "Hola!", "Adios!" };
        client.updateResourceProperty(name, values);
        
        client.queryResourceProperties(query, dialect);
        
        client.deleteResourceProperty(name);

        client.getResourceProperty(name);
    }
    
    public static void main(String[] args)
    {
        try
        {
            //
            // change these to point to different applications/servers
            //
            String webAppRoot = "/wsrf/services";
            int port = 8080;
                        
            //
            // test WSSG directory
            //
            
            String contextPath = webAppRoot + "/ServiceGroup";
            
            URI address = getLocalAddress(contextPath, port);
            EndpointReference epr = new EndpointReference(address);
            
            ServiceGroupClient sg = new ServiceGroupClient(epr);
            
            //
            // ping SG, wait a few seconds, then check for members 
            // that were added by the simulation factory
            //
            System.out.println("Pinging Muse to initialize the endpoint...");
            sg.getResourcePropertyDocument();
            System.out.println("Waiting 10 seconds while simulation creates WS-resources...");
            Thread.currentThread().sleep(10000);
            System.out.println("Done waiting - reading service group entries...");
            
            //
            // turn on tracing of SOAP messages
            //
            sg.setTrace(true);
            
            WsResourceClient[] members = sg.getMembers();
            
            System.out.println("Addresses of each service group entry:\n");
            
            for (int n = 0; n < members.length; ++n)
                System.out.println(members[n].getEndpointReference().getAddress());
            
            //
            // no ws-resources to test
            //
            if (members.length == 0)
                System.exit(0);

            //
            // test first WsResource we find
            //
            WsResourceClient client = null;
            
            for (int n = 0; n < members.length && client == null; ++n)
                if (members[n].getEndpointReference().getAddress().toString().endsWith("WsResource"))
                    client = members[n];
            
            client.setTrace(true);
            
            //
            // test WSRL operations
            //
            //destroy(client);
            //setTerminationTime(client);
            
            //
            // test WSRP read operations
            //
            getResourceProperty(client); 
            getMultipleResourceProperties(client);
            getResourcePropertyDocument(client);
                        
            //
            // test WSRP query operation
            //
            queryResourceProperties(client);
            
            //
            // test WSRP write operation
            //
            setResourceProperties(client);
        }
        
        catch (Throwable error)
        {
            error.printStackTrace();
        }
    }
}
