/*=============================================================================*
 *  Copyright 2006 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *=============================================================================*/

package org.apache.muse.test.http.impl;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;

import javax.xml.namespace.QName;

import org.apache.muse.util.MultiMap;
import org.apache.muse.ws.addressing.soap.SoapFault;
import org.apache.muse.ws.resource.impl.AbstractWsResourceCapability;

import org.apache.muse.test.http.HttpServer;
import org.apache.muse.test.http.StartFailedFault;
import org.apache.muse.test.http.StopFailedFault;
import org.apache.muse.test.http.SupportedLanguage;

public class HttpServerImpl 
    extends AbstractWsResourceCapability implements HttpServer
{    
    private String _installDir = null;
    
    private String _name = null;
    
    private int _port = -1;
    
    private SupportedLanguage[] _languages = null;
    
    public String getName()
    {
        return _name;
    }

    public int getPort()
    {
        return _port;
    }
    
    public QName[] getPropertyNames()
    {
        return PROPERTIES;
    }
    
    public SupportedLanguage[] getSupportedLanguage()
    {
        return _languages;
    }
    
    public void initialize() 
        throws SoapFault
    {
        super.initialize();
        
        _installDir = getInitializationParameter("httpd-install-dir");
        
        MultiMap configParams = null;
        
        try
        {
            configParams = readConfigFile(_installDir + "/conf/httpd.conf");
        }
        
        catch (IOException error)
        {
            throw new SoapFault("Error while reading httpd.conf.", error);
        }
        
        _name = readName(configParams);
        _port = readPort(configParams);
        _languages = readLanguages(configParams);
    }

    public void start() 
        throws StartFailedFault
    {
        try
        {
            Runtime.getRuntime().exec(_installDir + "/bin/httpd -k start");
            getLog().info("The httpd process was started successfully.");
        }
        
        catch (IOException error)
        {
            throw new StartFailedFault("The httpd process failed to start.");            
        }
    }

    public void stop() 
        throws StopFailedFault
    {
        try
        {
            Runtime.getRuntime().exec(_installDir + "/bin/httpd -k stop");
            getLog().info("The httpd process was stopped successfully.");
        }
        
        catch (IOException error)
        {
            throw new StopFailedFault("The httpd process failed to stop.");            
        }
    }
    
    protected MultiMap readConfigFile(String filePath)
        throws IOException
    {
        BufferedReader reader = new BufferedReader(new FileReader(filePath));
        String nextLine = null;
        
        MultiMap configParams = new MultiMap();
        
        while ((nextLine = reader.readLine()) != null)
        {
            nextLine = nextLine.trim();
            
            if (nextLine.length() == 0 || 
                nextLine.charAt(0) == '#' || 
                nextLine.charAt(0) == '<')
                continue;
            
            int space = nextLine.indexOf(' ');
            String name = nextLine.substring(0, space);
            String value = nextLine.substring(space + 1);
            configParams.put(name, value);
        }
        
        return configParams;
    }
    
    private SupportedLanguage[] readLanguages(MultiMap configParams)
    {
        Collection langStrings = (Collection)configParams.get("AddLanguage");
        
        //
        // no additinonal language support - default to english (en)
        //
        if (langStrings == null)
            return new SupportedLanguage[] { new SupportedLanguage("en", "en") };
        
        Iterator i = langStrings.iterator();
        
        SupportedLanguage[] languages = new SupportedLanguage[langStrings.size()];
        
        for (int n = 0; n < languages.length; ++n)
        {
            String next = (String)i.next();
            String[] tokens = next.split(" ");
            languages[n] = new SupportedLanguage(tokens[0], tokens[1]);
        }
        
        return languages;
    }
    
    private String readName(MultiMap configParams)
    {
        Collection nameStrings = (Collection)configParams.get("ServerName");
        
        if (nameStrings.isEmpty())
            return "httpd";
        
        String name = (String)nameStrings.iterator().next();
        
        int port = name.lastIndexOf(':');
        
        if (port >= 5) // don't get confused with : in http://
            name = name.substring(0, port);
        
        return name;
    }
    
    private int readPort(MultiMap configParams)
    {
        Collection portStrings = (Collection)configParams.get("Listen");
        String portString = (String)portStrings.iterator().next();
        return Integer.valueOf(portString).intValue();
    }
}
